var XWiki = (function (XWiki) {
// Start XWiki augmentation.
/**
 * Displays the title (tool tip) inside the text input when there's no value and the text input is not focused.
 */
XWiki.InputWithTitle = Class.create({
  initialize : function(input) {
    this.input = input;
    this.input.observe('focus', this._onFocus.bindAsEventListener(this));
    this.input.observe('blur', this._onBlur.bind(this));

    // We have to listen to the save event because 'Save & Continue' doesn't trigger a form submit event.
    document.observe('xwiki:class:save', this._onSave.bind(this));

    this._onBlur();
  },
  _onFocus : function(event) {
    if (this.input.hasClassName('unset')) {
      this.input.value = '';
      this.input.removeClassName('unset');
    }
  },
  _onBlur : function() {
    if (this.input.value == '') {
      this.input.addClassName('unset');
      this.input.value = this.input.title;
    }
  },
  _onSave : function() {
    this._onFocus();
    this._onBlur.bind(this).defer();
  }
});

/**
 * A text input that is automatically resized to fit its value.
 */
XWiki.AutoResizeInput = Class.create({
  initialize : function(input) {
    this.input = input;
    var resizeEvents = ['blur', 'change', 'cut', 'keyup', 'paste'];
    for(var i = 0; i < resizeEvents.length; i++) {
      this.input.observe(resizeEvents[i], this._resize.bindAsEventListener(this));
    }
    this.input.observe('keypress', this._reserveSpace.bindAsEventListener(this));

    this.meter = new Element('pre', {'class': 'autoResize-meter inherit'});
    this.meter.appendChild(document.createTextNode(''));
    this.input.addClassName('inherit').insert({after: this.meter});

    this._resize();
  },
  _resize : function(event) {
    if (event) {
      this._resize.bind(this).defer();
    } else {
      // Make sure the meter text is not set to the empty string because otherwise the meter offset width is wrongly
      // computed in Chrome. See http://code.google.com/p/chromium/issues/detail?id=105573 .
      this.meter.firstChild.data = this.input.value || ' ';
      this.input.setStyle({width: (this.meter.offsetWidth + 6) + 'px'});
    }
  },
  _reserveSpace : function(event) {
    /* Ignore: Backspace, Delete, End, Home, Left, Up, Right, Down. */
    var ignored = [8, 46, 35, 36, 37, 38, 39, 40];
    for(var i = 0; i < ignored.length; i++) {
      if (event.keyCode == ignored[i]) {
        return;
      }
    }
    this.input.setStyle({width: (this.input.offsetWidth + 6) + 'px'});
  }
});

/**
 * A toggle button backed by a select with two options.
 */
XWiki.ToggleSelect = Class.create({
  initialize : function(select) {
    this.select = select;
    this.select.hide();

    this.off = select.options[0];
    this.on = select.options[1];

    this.switcher = new Element('span', {'class': this.select.className, title: this.select.title});
    this.switcher.removeClassName('toggle').addClassName('toggle-switcher');
    this.switcher.appendChild(document.createTextNode(''));
    this.switcher.observe('click', this._toggle.bind(this));
    this.select.insert({after: this.switcher});

    this._update();
  },
  _toggle : function() {
    if (this.on.selected) {
      this.off.selected = true;
    } else {
      this.on.selected = true;
    }
    this._update();
  },
  _update : function() {
    this.switcher.firstChild.nodeValue = this.select.options[this.select.selectedIndex].text;
  }
});

// Allow widgets to catch the save event before the form is submitted when the 'Save & Continue' button is clicked. We
// need this hack because the 'Save & Continue' handler from actionbuttons.js is added right after the page is loaded,
// before any other code can register its own listener.
document.observe('xwiki:actions:save', function(event) {
  document.fire('xwiki:class:save', {originalEvent: event});
}.bindAsEventListener(window));
// End XWiki augmentation.
return XWiki;
}(XWiki || {}));
var XWiki = (function (XWiki) {
// Start XWiki augmentation.
/**
 * Handles the form field display (through AJAX), configure (toggle) and delete.
 */
XWiki.FormField = Class.create({
  initialize : function(container) {
    this.container = container;
  },
  enhance : function(data) {
    this.container.insert({top: this._createToolBox()});
    if (data) {
      this.container.addClassName('loading');
      new Ajax.Request(data, {
        method: 'get',
        onSuccess: this._display.bind(this)
      });
    } else {
      this._display(null);
    }
  },
  getContainer : function() {
    return this.container;
  },
  getViewer : function() {
    return this.container.down('.field-viewer')
  },
  getConfig : function() {
    return this.container.down('.field-config');
  },
  getToolBox : function() {
    return this.container.down('.toolBox');
  },
  getName : function() {
    if (!this.container._typeInput) {
      // Field type should be stored in the first hidden input under the field container.
      this.container._typeInput = this.container.down('input[type=hidden]');
    }
    return this.container._typeInput.name.substring('type-'.length);
  },
  getPropertyId : function(propertyName) {
    return 'field-' + this.getName() + '_' + propertyName;
  },
  _display : function(response) {
    // If the field container is still attached, update its content.
    if (this.container.parentNode) {
      if (response) {
        this.container.removeClassName('loading');
        this._injectHTML(response.responseText);
      }

      this.getToolBox().insert({top: this._createConfigIcon()});
      this._onConfigure();

      document.fire('xwiki:class:displayField', {field: this, 'new': !!response});

      this._twoColumnConfigDisplay();
    }
  },
  _injectHTML : function(html) {
    // We don't use Element#update() because it doesn't move external scripts and sheets into HEAD and also because we
    // don't want to support in-line scripts in displayers.
    var container = new Element('div');
    container.innerHTML = html;
    var head = document.body.previous('head');
    container.select('link').each(function(link) {
      head.insert(link);
    });
    container.select('script').each(function(script) {
      if (script.src) {
        head.insert(new Element('script', {type: script.type, src: script.readAttribute('src')}));
      }
      script.remove();
    });
    this.container.insert(container);
  },
  _createToolBox : function() {
    var deleteIcon = new Element('img', {
      src: '../../attachment/xwiki/AppWithinMinutes/ClassEditSheet/bulletcross.png',
      alt: '\u0423\u0434\u0430\u043B\u0438\u0442\u044C',
      title: '\u0423\u0434\u0430\u043B\u0438\u0442\u044C'
    });
    deleteIcon.observe('click', this._onDelete.bindAsEventListener(this));

    var toolBox = new Element('div', {'class': 'toolBox'});
    toolBox.insert(deleteIcon);
    return toolBox;
  },
  _createConfigIcon : function() {
    return new Element('img').observe('click', this._onConfigure.bindAsEventListener(this));
  },
  _onConfigure : function(event) {
    var icon = event ? event.element() : this.getToolBox().down('img');
    var fieldConfig = this.getConfig();
    if (fieldConfig.visible()) {
      fieldConfig.hide();
      icon.alt = icon.title = '\u041D\u0430\u0441\u0442\u0440\u043E\u0439\u043A\u0430';
      icon.src = '../../resources/icons/silk/bullet_wrench.png';
      this._maybePreview();
    } else {
      fieldConfig.show();
      icon.alt = '\u041F\u0440\u0435\u0434\u043F\u0440\u043E\u0441\u043C\u043E\u0442\u0440';
      icon.title = '\u0421\u043A\u0440\u044B\u0442\u044C \u043F\u0430\u043D\u0435\u043B\u044C \u043D\u0430\u0441\u0442\u0440\u043E\u0439\u043A\u0438 \u0438 \u043F\u0440\u0438\u043C\u0435\u043D\u0438\u0442\u044C \u0438\u0437\u043C\u0435\u043D\u0435\u043D\u0438\u044F';
      icon.src = '../../attachment/xwiki/AppWithinMinutes/ClassEditSheet/bullettick.png';
      this._maybeSavePreviewData();
    }
  },
  _onDelete : function(event) {
    new XWiki.widgets.ConfirmationBox({
      onYes : this._onConfirmedDelete.bind(this)
    }, {
      confirmationText : 'Вы уверены, что хотите удалить данное свойство?'
    });
  },
  _onConfirmedDelete : function() {
    this.container.remove();
    document.fire('xwiki:class:deleteField', {field: this});
  },
  _maybeSavePreviewData : function() {
    // Initialize the preview data.
    this.previewData || this._savePreviewData();
    // Make sure the preview data is updated when the field is renamed.
    if (!this._renameListener) {
      this._renameListener = function(event) {
        if (event.memo.field.getContainer() == this.container) {
          this._savePreviewData();
        }
      }.bindAsEventListener(this);
      document.observe('xwiki:class:renameField', this._renameListener);
    }
  },
  _savePreviewData : function() {
    this.previewData = this._asHash();
    // Ignore properties that don't affect the preview.
    ['name', 'prettyName', 'hint', 'required', 'number'].each(function(propertyName) {
      this.previewData.unset(this.getPropertyId(propertyName));
    }.bind(this));
    // Add parameters required by the AJAX preview request.
    this.previewData.set('preview', true);
    this.previewData.set('xpage', 'plain');
  },
  _maybePreview : function() {
    if(!this.previewData) {
      return;
    }
    // Check if there are configuration changes.
    var data = new Hash();
    var areEqual = function(alice, bob) {
      if (typeof alice == typeof bob) {
        if (alice.join) {
          return alice.join() == bob.join();
        }
        return alice == bob;
      }
      return false;
    }
    this._asHash().each(function(pair) {
      var oldValue = this.previewData.get(pair.key);
      if (oldValue != undefined && !areEqual(pair.value, oldValue)) {
        data.set(pair.key, pair.value);
      }
    }.bind(this));
    var modifiedKeys = data.keys();
    if (modifiedKeys.length > 1 || (modifiedKeys.length == 1 && !modifiedKeys[0].endsWith('_0_' + this.getName()))) {
      this._preview(this.previewData.clone().update(data));
    }
  },
  _preview : function(data) {
    var preview = this.getViewer().down('dd');
    if (preview) {
      preview.addClassName('loading');
      // Firefox 3.6 doesn't resolve the empty string to the current page URL so we have to explicitly specify it.
      new Ajax.Request(window.location.href, {
        parameters: data.toQueryString(),
        onSuccess: function(response) {
          preview.removeClassName('loading');
          preview.update(response.responseText);
          this.previewData = data;
          document.fire('xwiki:class:previewField', {field: this});
          document.fire('xwiki:dom:updated', {elements: [preview]});
        }.bind(this)
      });
    }
  },
  _asHash : function() {
    // Simulate a submit so that we collect the right values.
    document.fire('xwiki:class:save', {originalEvent: {memo: {}}});
    var inputs = this.container.select('input', 'textarea', 'select');
    return new Hash(Form.serializeElements(inputs, {hash: true}));
  },
  _twoColumnConfigDisplay : function() {
    var leftColumn = this.getConfig();
    var fieldConfig = new Element('div', {'class': leftColumn.className});
    leftColumn.insert({before: fieldConfig});
    fieldConfig.insert(leftColumn);
    fieldConfig.hide();
    leftColumn.className = 'leftColumn';
    leftColumn.show();
    var rightColumn = new Element('dl', {'class': 'rightColumn'});
    fieldConfig.insert(rightColumn);

    var visibleDTs = [];
    leftColumn.select('dt').each(function(dt) {
      if(dt.visible()) {
        visibleDTs.push(dt);
      }
    });

    var half = Math.ceil(visibleDTs.length / 2);
    for(var i = half; i < visibleDTs.length; i++) {
      var dt = visibleDTs[i];
      var dd = dt.next();
      rightColumn.insert(dt);
      if (dd && dd.tagName.toLowerCase() == 'dd') {
        rightColumn.insert(dd);
      }
    }

    fieldConfig.insert(new Element('div', {'style': 'clear: left'}));
  }
});

/**
 * Manages the form field names. Generates unique names for newly added fields and renames field configuration property ids/names when a field is renamed.
 */
XWiki.FormFieldNameManager = Class.create({
  initialize : function() {
    // Generate a unique field name when a new field is dragged from the palette to the form canvas.
    document.observe('xwiki:class:displayField', this._maybeGenerateFieldName.bind(this));
    // Rename the field configuration property ids/names when a field is renamed. This is needed for AJAX Save & Continue.
    document.observe('xwiki:document:saved', this._maybeRenameFields.bind(this));
  },
  _maybeGenerateFieldName : function(event) {
    if (!event.memo['new']) {
      return;
    }
    var field = event.memo.field;
    var nameInput = $(field.getPropertyId('name'));
    var counter = 1;
    // We check the type meta property because there can be unsupported field types for which we know only their type.
    while($('type-' + nameInput.value + counter++));
    nameInput.value = nameInput.value + (counter - 1);
    this._maybeRenameField(field);
  },
  _maybeRenameFields : function() {
    $('fields').childElements().each(function(item) {
      this._maybeRenameField(new XWiki.FormField(item));
    }.bind(this));
  },
  _maybeRenameField : function(field) {
    var oldName = field.getName();
    var newName = $(field.getPropertyId('name')).value;
    if (oldName != newName) {
      this._maybeRenameElement(oldName, newName, field.getContainer());
      field.getContainer().descendants().each(this._maybeRenameElement.bind(this, oldName, newName));
      document.fire('xwiki:class:renameField', {field: field});
    }
  },
  _maybeRenameElement : function(oldName, newName, element) {
    ['id', 'name', 'for'].each(function(attribute) {
      var value = element.readAttribute(attribute) || '';
      if (value.startsWith('xwiki-form-' + oldName)) {
        // This is for fields with multiple values (e.g. radio buttons).
        value = 'xwiki-form-' + newName + value.substring(11 + oldName.length);
      } else if (value.endsWith('_0_' + oldName)) {
        // This is for field default value.
        value = value.substring(0, value.length - oldName.length) + newName;
      } else {
        // This is for field meta properties.
        var nameIndex = value.indexOf('-') + 1;
        if (nameIndex > 0 && (value.substring(nameIndex) == oldName || value.substring(nameIndex).startsWith(oldName + '_'))) {
          value = value.substring(0, nameIndex) + newName + value.substring(nameIndex + oldName.length);
        }
      }
      if (value != '') {
        element.writeAttribute(attribute, value);
      }
    });
  }
});

/**
 * Enhances the way the field pretty name is edited. Allows users to edit the pretty name in-place inside the field preview.
 */
XWiki.FormFieldPrettyNameManager = Class.create({
  initialize : function() {
    document.observe('xwiki:class:displayField', this._onDisplayField.bindAsEventListener(this));
  },
  _onDisplayField : function(event) {
    var field = event.memo.field;
    var prettyNameInput = $(field.getPropertyId('prettyName'));
    // Hide the DD containing the pretty name input.
    var parentDD = prettyNameInput.up('dd');
    parentDD.hide();
    // Hide the DT containing the pretty name input label.
    var previousDT = parentDD.previous('dt');
    previousDT.hide();
    // Use the label text as tool tip for the pretty name input, if the title attribute is not specified.
    if (!prettyNameInput.title) {
      var label = previousDT.down('label');
      prettyNameInput.title = label.textContent || label.innerText;
    }
    // Replace the preview label with the pretty name input.
    var previewLabel = field.getViewer().down('label');
    previewLabel.writeAttribute('for', '');
    previewLabel.removeChild(previewLabel.lastChild);
    previewLabel.insert(prettyNameInput);
    // Enhance the pretty name input.
    new XWiki.InputWithTitle(prettyNameInput);
    new XWiki.AutoResizeInput(prettyNameInput);
  }
});

/**
 * Adds a tool tip to the default value input.
 */
XWiki.FormFieldDefaultValueManager = Class.create({
  initialize : function() {
    var enhanceFieldDefaultValue = this._enhanceFieldDefaultValue.bindAsEventListener(this);
    document.observe('xwiki:class:displayField', enhanceFieldDefaultValue);
    document.observe('xwiki:class:previewField', enhanceFieldDefaultValue);

    // We need to listen to save event to rename the default value inputs to match the new field name.
    document.observe('xwiki:class:save', this._onSave.bind(this));
  },
  _enhanceFieldDefaultValue : function(event) {
    var field = event.memo.field;
    var fieldName = field.getName();
    field.getViewer().select('input[type=text]', 'textarea').each(function(input) {
      if (!input.title && input.name.endsWith('_0_' + fieldName)) {
        input.title = '\u0417\u043D\u0430\u0447\u0435\u043D\u0438\u0435 \u043F\u043E \u0443\u043C\u043E\u043B\u0447\u0430\u043D\u0438\u044E';
      }
    });
  },
  _onSave : function() {
    $('fields').childElements().each(function(item) {
      this._maybeRenameDefaultValue(new XWiki.FormField(item));
    }.bind(this));
  },
  _maybeRenameDefaultValue : function(field) {
    var oldName = field.getName();
    var newName = $(field.getPropertyId('name')).value;
    if(oldName != newName) {
      field.getViewer().select('input', 'textarea', 'select').each(function(element) {
        if (element.name.endsWith('_0_' + oldName)) {
          element.name = element.name.substring(0, element.name.length - oldName.length) + newName;
        }
      });
    }
  }
});

/**
 * Makes the hint editable in place.
 */
XWiki.FormFieldHintManager = Class.create({
  initialize : function() {
    document.observe('xwiki:class:displayField', this._onDisplayField.bindAsEventListener(this));
  },
  _onDisplayField : function(event) {
    var field = event.memo.field;
    var hintInput = $(field.getPropertyId('hint'));
    // TODO: Remove this test after the 'hint' meta property is added to the platform.
    if (hintInput) {
      hintInput.addClassName('xHint');
      if (!hintInput.title) {
        hintInput.title = 'Hint';
      }
      // Move the hint input below the pretty name input, in the field viewer.
      var dd = hintInput.up('dd');
      var dt = dd.previous('dt');
      field.getViewer().down('label').insert({after: hintInput});
      dt.remove();
      dd.remove();
      // Enhance the hint input.
      new XWiki.InputWithTitle(hintInput);
      new XWiki.AutoResizeInput(hintInput);
    }
  }
});

/**
 * Replaces the input of the "required" field property with a toggle button.
 */
XWiki.FormFieldRequiredManager = Class.create({
  initialize : function() {
    document.observe('xwiki:class:displayField', this._onDisplayField.bindAsEventListener(this));
  },
  _onDisplayField : function(event) {
    var field = event.memo.field;
    var requiredInput = $(field.getPropertyId('required'));
    // TODO: Remove this test after the 'required' meta property is added to the platform.
    if (requiredInput) {
      var prettyNameLabel = field.getViewer().down('label');
      var wrapper = new Element('div', {'class': 'labelLine'});
      prettyNameLabel.insert({before: wrapper});
      wrapper.insert(prettyNameLabel);
      wrapper.insert('<label><select class="xRequired" title="Click to toggle between (Optional) and (Required)"><option>(Optional)</option><option>(Required)</option></select></label>');
      new XWiki.ToggleSelect(prettyNameLabel.next().down('select'));
    }
  }
});

/**
 * Manages the sortable list of form fields. Accepts fields dragged from the field palette.
 */
XWiki.FormCanvas = Class.create({
  initialize : function(container) {
    this.container = $(container);
    Droppables.add(this.container, {accept: 'field', hoverclass: 'hovered', onDrop: this._onDrop.bind(this)});

    document.observe('xwiki:class:deleteField', this._onDeleteField.bind(this));
    document.observe('xwiki:class:displayField', this._onDisplayField.bindAsEventListener(this));

    this.fields = this.container.down('ul');
    if (!this.fields) {
      this.fields = new Element('ul');
      this.container.insert(this.fields);
    }
    this.fields.id = 'fields';
    if (this.fields.empty()) {
      this.container.addClassName('empty');
    } else {
      this._enhanceFields();
    }
  },
  _enhanceFields : function() {
    this.fields.childElements().each(function(item) {
      new XWiki.FormField(item).enhance();
    });
  },
  _onDrop : function(field) {
    var fieldContainer = new Element('li', {'data-new': 'true'});
    this.fields.insert(fieldContainer);
    this.container.removeClassName('empty');
    new XWiki.FormField(fieldContainer).enhance(field.down('.data').value);
  },
  _onDeleteField : function() {
    if (this.fields.empty()) {
      this.container.addClassName('empty');
    } else {
      this._updateOrder();
    }
  },
  _onDisplayField : function(event) {
    var field = event.memo.field;

    // We have to set the ID of the field container as otherwise Sortable.onUpdate() isn't triggered.
    field.getContainer().id = 'field-' + field.getName();

    // Hide the property number, as ordering can be done by drag and drop
    var numberInput = $(field.getPropertyId('number'));
    numberInput.up().hide();
    numberInput.up().previous().hide();

    // Create and insert the move icon.
    var moveIcon = new Element('img', {
      src: '../../attachment/xwiki/AppWithinMinutes/ClassEditSheet/arrow-move.png',
      alt: '\u041F\u0435\u0440\u0435\u043C\u0435\u0441\u0442\u0438\u0442\u044C',
      title: '\u041F\u0435\u0440\u0435\u0442\u044F\u0433\u0438\u0432\u0430\u0439\u0442\u0435 \u0437\u043D\u0430\u0447\u0435\u043D\u0438\u044F \u0434\u043B\u044F \u0438\u0437\u043C\u0435\u043D\u0435\u043D\u0438\u044F \u0438\u0445 \u043F\u043E\u0440\u044F\u0434\u043A\u0430',
      'class': 'icon-move'
    });
    field.getToolBox().insert({bottom: moveIcon});

    Sortable.create('fields', {
      handle: 'icon-move',
      format: /^[^_\-](?:[A-Za-z0-9\-\_]*)[\-](.*)$/,
      onUpdate : this._updateOrder.bind(this)
    });

    if (event.memo['new']) {
      this._updateOrder();
    }
  },
  _updateOrder : function() {
    var items = this.fields.childElements();
    for(var i = 0; i < items.length; i++) {
      var field = new XWiki.FormField(items[i]);
      if (field.getConfig()) {
        $(field.getPropertyId('number')).value = i + 1;
      } else {
        // Stop updating the order. This method will be called again when the field will be loaded.
        break;
      }
    }
  }
});

/**
 * Manages the palette of form fields.
 */
XWiki.FormFieldPalette = Class.create({
  initialize : function(container) {
    $(container).select('.field').each(function(field) {
      new Draggable(field, {
        revert: true,
        reverteffect: function(field) {
          // Reset the in-line style.
          field.setStyle({
            height: '',
            left: '',
            position: '',
            top: '',
            zIndex: '',
            width: ''
          });
        },
        ghosting: true
      });
    });
  }
});
// End XWiki augmentation.
return XWiki;
}(XWiki || {}));

(function() {
  function init() {
    var form = $('inline');
    if (!form) {
      return false;
    }
    // Let the sheet handle the form submit. The form is submitted by default to the preview action which dispatches the
    // request to the save action if the save button is detected on the request parameters. By submitting to the edit
    // action the edit sheet is evaluated and thus it can handle the save by itself.
    form.action = XWiki.currentDocument.getURL('edit');

    // Apply the vertical form layout standard.
    form.addClassName('xform');

    // Remove the preview button.
    form.down("input[name='action_preview']").remove();

    // Rename the save buttons.
    form.down("input[name='action_save']").name = 'xaction_save';
    form.down("input[name='action_saveandcontinue']").name = 'xaction_saveandcontinue';

    // Make sure the 'Save & Continue' button is submitted when clicked.
    document.observe('xwiki:class:save', function(event) {
      event = event.memo.originalEvent;
      if (event.memo['continue']) {
        var submitButton = event.memo.originalEvent.element();
        submitButton.insert({after: new Element('input', {type: 'hidden', 'name': submitButton.name})});
        (function () {
          submitButton.next().remove();
        }).defer();
      }
    }.bindAsEventListener(window));

    // Be prepared to enhance some of the field properties.
    new XWiki.FormFieldNameManager();
    new XWiki.FormFieldPrettyNameManager();
    new XWiki.FormFieldHintManager();
    new XWiki.FormFieldDefaultValueManager();
    new XWiki.FormFieldRequiredManager();

    // Make the palette and the canvas live.
    new XWiki.FormFieldPalette('palette');
    new XWiki.FormCanvas('canvas');

    return true;
  }
  (XWiki.domIsLoaded && init()) || document.observe('xwiki:dom:loaded', init);
}).call();

require(['jquery', 'xwiki-events-bridge'], function($) {
  $(document).on('xwiki:class:displayField xwiki:class:previewField', function(event, data) {
    var container = $(data.field.getContainer());
    if (container.attr('data-new') === 'true') {
      // We can't suggest property values for properties that don't exist yet (have not been saved) so we provide
      // suggestions for the template property that was used to create them. Note that the suggested values depend on
      // the saved property meta data so changing the property (field) meta data may not affect the suggestions until
      // those changes are saved.
      var templateHiddenInput = container.find('#' + 'template-' + data.field.getName());
      var propertyValueSuggester = container.find('.field-viewer .suggest-propertyValues').first();
      propertyValueSuggester.attr({
        'data-className': templateHiddenInput.val(),
        'data-propertyName': templateHiddenInput.attr('data-propertyName')
      });
    }
  }).on('xwiki:document:saved', function(event) {
    // We need to update the property value suggesters because:
    // * newly saved properties should have their own suggestions instead of relying on the template property
    // * for renamed properties we need to fetch the suggestions from a different location
    $('ul#fields > li').each(function() {
      var container = $(this);
      container.removeAttr('data-new');
      var propertyValueSuggester = container.find('.field-viewer .suggest-propertyValues').first();
      if (propertyValueSuggester.length > 0) {
        // We need to preserve the selected values because they are lost when the suggester is destroyed.
        var selectedValues = propertyValueSuggester.children();
        propertyValueSuggester[0].selectize.destroy();
        // Restore the selected values.
        propertyValueSuggester.empty().append(selectedValues);
        var className = XWiki.Model.serialize(XWiki.currentDocument.documentReference.relativeTo(
          new XWiki.WikiReference(XWiki.currentWiki)));
        propertyValueSuggester.attr({
          'data-className': className,
          'data-propertyName': propertyValueSuggester.attr('name').substr((className + '_0_').length)
        });
        propertyValueSuggester.suggestPropertyValues();
      }
    });
  });
});
